#!/bin/bash

set -e

INSTALL_DIR="/opt/voiceforge"
SERVICE_NAME="voiceforge"
SERVICE_USER="voiceforge"

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

check_root() {
    if [[ $EUID -ne 0 ]]; then
        log_error "This script must be run as root (use sudo)"
        exit 1
    fi
}

confirm_uninstall() {
    echo -e "${YELLOW}"
    echo "╔═══════════════════════════════════════════════════════════╗"
    echo "║                      ⚠ WARNING ⚠                         ║"
    echo "╠═══════════════════════════════════════════════════════════╣"
    echo "║                                                           ║"
    echo "║  This will completely remove VoiceForge including:        ║"
    echo "║                                                           ║"
    echo "║  • Application files in /opt/voiceforge                   ║"
    echo "║  • Systemd service                                        ║"
    echo "║  • Service user account                                   ║"
    echo "║                                                           ║"
    echo "║  FFmpeg and Node.js will NOT be removed.                  ║"
    echo "║                                                           ║"
    echo "╚═══════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
    
    read -p "Are you sure you want to uninstall VoiceForge? (yes/no): " confirm
    
    if [[ "$confirm" != "yes" ]]; then
        log_info "Uninstall cancelled."
        exit 0
    fi
}

stop_service() {
    if systemctl is-active --quiet ${SERVICE_NAME}; then
        log_info "Stopping service..."
        systemctl stop ${SERVICE_NAME}
    fi
}

remove_service() {
    if [[ -f /etc/systemd/system/${SERVICE_NAME}.service ]]; then
        log_info "Disabling service..."
        systemctl disable ${SERVICE_NAME} 2>/dev/null || true
        
        log_info "Removing service file..."
        rm -f /etc/systemd/system/${SERVICE_NAME}.service
        
        log_info "Reloading systemd..."
        systemctl daemon-reload
    fi
}

remove_files() {
    if [[ -d "$INSTALL_DIR" ]]; then
        log_info "Removing application files..."
        rm -rf "$INSTALL_DIR"
    fi
    
    for backup in /opt/voiceforge_backup_*; do
        if [[ -d "$backup" ]]; then
            log_info "Removing backup: $backup"
            rm -rf "$backup"
        fi
    done
}

remove_user() {
    if id "$SERVICE_USER" &>/dev/null; then
        log_info "Removing service user..."
        userdel "$SERVICE_USER" 2>/dev/null || true
    fi
}

main() {
    echo -e "${RED}"
    echo "╔═══════════════════════════════════════════════════════════╗"
    echo "║          VoiceForge Uninstall Script                      ║"
    echo "╚═══════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
    
    check_root
    confirm_uninstall
    stop_service
    remove_service
    remove_files
    remove_user
    
    echo ""
    echo -e "${GREEN}✓ VoiceForge has been completely uninstalled.${NC}"
    echo ""
    echo -e "${BLUE}Note: FFmpeg and Node.js were not removed.${NC}"
    echo -e "${BLUE}To remove them manually:${NC}"
    echo -e "${BLUE}  Ubuntu/Debian: apt remove ffmpeg nodejs${NC}"
    echo -e "${BLUE}  CentOS/RHEL:   yum remove ffmpeg nodejs${NC}"
    echo ""
}

main "$@"
